<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration {
    public function up(): void
    {
        // LAND OWNERS
        Schema::create('landloards', function (Blueprint $table) {
            $table->id();
            $table->string('name');
            $table->string('cnic')->nullable();
            $table->string('phone')->nullable();
            $table->string('email')->nullable();
            $table->text('address')->nullable();
            $table->boolean('is_active')->default(true);
            $this->addRefrenceFields($table);
            $this->addAuditFields($table);
            $table->timestamps();
        });

        // PROJECTS // Schemes
        Schema::create('projects', function (Blueprint $table) {
            $table->id();
            $table->date('entry_date')->nullable();
            $table->string('manual_no')->nullable()->comment('Manual number for the project');
            $table->string('name')->unique();
            $table->string('known_as')->nullable();
            $table->string('type')->comment('Housing Society, Shopping Mall, Office Building, etc.');
            $table->foreignId('landloard_id')->nullable()->constrained('landloards')->onDelete('set null');
            $table->decimal('land_cost', 15, 2)->nullable();
            // $table->decimal('amount_paid', 15, 2)->default(0);
            // $table->decimal('amount_due', 15, 2)->default(0);
            $table->text('address')->nullable();
            $table->integer('completion_year')->nullable();
            $table->integer('square_feet_per_marla')->default(0);
            $table->text('current_phase')->nullable();
            $this->addRefrenceFields($table);
            $this->addAuditFields($table);
            $table->timestamps();
        });

        // PROPERTY TYPE
        Schema::create('property_types', function (Blueprint $table) {
            $table->id();
            $table->string('title')->unique();
            $this->addRefrenceFields($table);
            $this->addAuditFields($table);
            $table->timestamps();
        });

        // PROPERTIES
        Schema::create('properties', function (Blueprint $table) {
            $table->id();
            $table->date('entry_date')->nullable();
            $table->foreignId('property_type_id')->nullable()->constrained('property_types')->onDelete('set null');
            $table->foreignId('project_id')->nullable()->constrained('projects')->onDelete('set null')->comment('Scheme Title');
            $table->text('sector_title')->nullable();
            $table->string('property_no')->nullable();
            $table->string('title');
            $table->boolean('is_price_negotiable')->default(false); 
            $table->string('booking_no')->default('Un Booked');
            $table->decimal('area', 10, 2)->default(0.00)->comment('Area in sqft');
            $table->decimal('price', 15, 2)->nullable();
            $table->text('attribuites')->nullable()->comment('comma seperated attributes for additional property details');
            $this->addRefrenceFields($table);
            $this->addAuditFields($table);
            $table->timestamps();
        });

        // CUSTOMERS
        Schema::create('customers', function (Blueprint $table) {
            $table->id();
            $table->string('name');
            $table->string('cnic')->nullable();
            $table->string('passport')->nullable();
            $table->string('dob')->nullable();
            $table->string('phone')->nullable();
            $table->string('email')->nullable();
            $table->text('address')->nullable();
            $this->addRefrenceFields($table);
            $this->addAuditFields($table);
            $table->timestamps();
        });

        // INSTALLMENT PLANS
        Schema::create('bookings', function (Blueprint $table) {
            $table->id();
            $table->foreignId('property_id')->nullable()->constrained('properties')->onDelete('set null');
            $table->foreignId('customer_id')->nullable()->constrained('customers')->onDelete('set null')->comment('Party Name');
            $table->date('date')->nullable();
            $table->string('manual_booking_no')->nullable();
            $table->integer('booking_no')->nullable();
            $table->string('from_bvn_no')->nullable();
            $table->string('salesman')->nullable();
            $table->double('salesman_commission')->default(0);
            $table->string('dealer')->nullable();
            $table->string('next_to_kin')->nullable();
            $table->string('status')->nullable()->comment('active, cancelled, completed');
            $table->boolean('is_verified')->nullable();
            $table->decimal('measurement_charges', 15, 2);
            $table->decimal('total_price', 15, 2);
            $table->decimal('down_payment', 15, 2)->default(0);
            $table->decimal('installment_amount', 15, 2);
            $table->date('start_date');
            $this->addRefrenceFields($table);
            $this->addAuditFields($table);
            $table->timestamps();
        });

        // INSTALLMENTS
        Schema::create('installments', function (Blueprint $table) {
            $table->id();
            $table->foreignId('booking_id')->nullable()->constrained('bookings')->onDelete('set null');
            $table->integer('installment_number')->nullable();
            $table->string('installment_type')->comment('monthly, quarterly, yearly, custom');
            $table->date('due_date');
            $table->date('reminder_date')->nullable();
            $table->decimal('amount', 15, 2);
            $table->decimal('amount_paid', 15, 2)->default(0);
            $table->date('paid_at')->nullable();
            $table->string('status')->nullable()->comment('pending, paid, overdue');
            $this->addRefrenceFields($table);
            $this->addAuditFields($table);
            $table->timestamps();
        });

        Schema::create('customer_payments', function (Blueprint $table) {
            $table->id();
            $table->foreignId('booking_id')->nullable()->constrained('bookings')->onDelete('set null');
            $table->foreignId('customer_id')->nullable()->constrained('customers')->onDelete('set null');
            $table->date('payment_date');
            $table->decimal('amount', 15, 2)->default(0);
            $table->string('payment_method')->nullable();
            $table->string('transaction_reference')->nullable();
            $this->addRefrenceFields($table);
            $this->addAuditFields($table);
            $table->timestamps();
        });

        Schema::create('landloard_payment_schedule', function (Blueprint $table) {
            $table->id();
            $table->foreignId('landloard_id')->nullable()->constrained('landloards')->onDelete('set null');
            
            $table->integer('manual_number')->nullable();
            $table->integer('schedule_number')->nullable();
            $table->string('session')->nullable();
            $table->string('installment_type')->comment('monthly, quarterly, yearly, custom');
            $table->date('due_date');
            $table->date('reminder_date')->nullable();
            $table->decimal('amount', 15, 2)->comment('Amount to be paid to landloard');
            $table->decimal('amount_paid', 15, 2)->default(0);
            $table->date('paid_at')->nullable();
            $table->string('status')->nullable()->comment('pending, paid, overdue');
            $this->addRefrenceFields($table);
            $this->addAuditFields($table);
            $table->timestamps();
        });

        Schema::create('landloard_payments', function (Blueprint $table) {
            $table->id();
            $table->foreignId('landloard_id')->nullable()->constrained('landloards')->onDelete('set null');
            $table->date('payment_date');
            $table->decimal('amount_paid', 15, 2);
            $table->string('payment_method')->nullable();
            $table->string('transaction_reference')->nullable();
            $this->addRefrenceFields($table);
            $this->addAuditFields($table);
            $table->timestamps();
        });
    }

    private function addRefrenceFields(Blueprint $table): void
    {
        $table->text('description')->nullable();
        $table->string('reference_no')->unique()->nullable()->comment('Edusoft reference number for the land owner');
    }

    private function addAuditFields(Blueprint $table): void
    {
        $table->unsignedBigInteger('created_by')->nullable();
        $table->unsignedBigInteger('updated_by')->nullable();
        $table->string('source_ip')->nullable();
        $table->string('source_mac')->nullable();
    }

    public function down(): void
    {
        Schema::dropIfExists('installments');
        Schema::dropIfExists('customer_payments');
        Schema::dropIfExists('bookings');
        Schema::dropIfExists('customers');
        Schema::dropIfExists('properties');
        Schema::dropIfExists('property_types');
        Schema::dropIfExists('projects');
        Schema::dropIfExists('landloard_payment_schedule');
        Schema::dropIfExists('landloard_payments');
        Schema::dropIfExists('landloards');
    }
};